<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly 
define('FOXPOST_PHP_API_VERSION', '1.1.8');

class FoxPostApi{
	const NAME		= 'FoxPostApi';
	const APIURL	= 'https://webapi.foxpost.hu/api/';           
	const WEBURL	= 'https://foxpost.hu/belepes';
	const TRACEURL  = 'https://foxpost.hu/csomagkovetes/?code=%s';
	const INFOURL	= 'https://foxpost.hu/ugyfelszolgalat';

    /**                                                
     * APM list
     *
     * @var string
     *
     * @since 1.1.3
     */    
	const APMJSON	= 'https://cdn.foxpost.hu/foxpost_terminals_extended_v3.json';
    
	private $username = '';      
	private $password = '';
	private $apikey = '';
    private $log_level = 2;
    private $eua = '';
    
	//public $endpoints_methods = array(
    const ENDP_METHODS = [
		'post' => array('parcel', 'tracks', 'label'),  
		'get' => array('address','tracking') 
	];
	//public $endpoints_data_req = array(
    const ENDP_DATA_REQ = [
		'address' => array(),  
		'parcel' => array('recipientEmail', 'recipientName', 'recipientPhone', 'size', 'refCode'),  
		'tracking' => array('barcode'),  
		'label' => array('barcode'),  
	];    
	//public $endpoints_data_resp_type = array(
    const ENDP_RESP_TYPE = [
		'json' => array('tracking', 'tracking/tracks', 'parcel', 'address'),  
		'pdf' => array('label'),  
	];    
        
    
    public $error_msg   = '';
    public $error_msgs  = array();
    public $error       = false;
    private $logs       = array();
    private $logfile    = '/tmp/FoxPostApi.log';
    private $tmp_dir    = '/tmp/FoxPostApi/';
    private $options    = array();
    private $name       = '';

    /**
     * 
     * 
     * @param $log_level int Set API log level
     * @return 
     */ 
	public function __construct($log_level=2){                                          
        $this->name = self::NAME;
        $this->log_level = $log_level;
        
        if (!$this->_isCurlAvailable()) $this->_err('CURL_NOT_INSTALLED');
	}
    
    
    public function setConfig($options){
        $this->options = $options;    
    }
    
    /**
     * ReSet
     * 
     */ 
    public function reset(){
        $this->error_msg    = '';
        $this->error        = false;
        $this->logs         = array();
        return true; 
    }
        
    /**
     * Set API logon information. Sometimes not needed.
     * 
     */ 
    public function setCred($username, $password, $apikey){
        if ( empty($username) ||  empty($password) || empty($apikey) ) $this->_err('CRED_NOT_SET');
        $this->_log("setCred: $username", 5);	
        $this->username = $username;
        $this->password = $password; 
        $this->apikey   = $apikey;    
        return true; 
    }
    
    /**
     * 
     * 
     * @param $format string
     */ 
    public function getLog($format='string'){
		switch ( $format ) {
				case 'string':
                    return implode("", $this->logs);
					break;
				case 'array':
                    return $this->logs;
					break;
				default:
                    return "";
		}    
    }
    
    
    /**
     * Please log this, if is false dont log to file
     * 
     * @param $logfilelocation string|false
     */ 
    public function setLogFile($logfilelocation=false){
        if ($logfilelocation==false){
            $this->logfile = false; 
            return true;
        }     
    }

    /**
     * Set tempolary folder for logs&labels
     * 
     * @param $directory string
     */         
    public function setTmpDir($directory='/tmp/'){
        global $wp_filesystem;

        $wp_filesystem->chmod($directory, 0755);
                    
        if ( !is_dir($directory . 'labels') ) $wp_filesystem->mkdir($directory . 'labels', 0755, true);
        if ( !is_dir($directory . 'logs') )   $wp_filesystem->mkdir($directory . 'logs'  , 0755, true);

        if ( !$wp_filesystem->is_writable($directory) ) $this->_err("Directory $directory not writeable!");
       
        $this->tmp_dir = $directory;
        
        $this->_log( "setTmpDir: $directory", 6);	
        
        return true;  
    }
    
    /**
     * Login test
     * 
     */ 
	public function login(){
        $this->_log( "login", 5);	       
        if ( $this->parcelAddress()!==false ) return true;
		return false;		
	}
    
    /**                                        
     * Get Address from Foxpost
     * 
     */ 
	public function parcelAddress(){
        global $wp_filesystem;
        $this->reset();
        
        $this->_log( "parcelAddress", 5);	
        $response = $this->_getResponse( 'address' );
        if ($response==false || empty($response)) return false;
        
        $filename = 'return-addresses.json'; 
        $a = $wp_filesystem->put_contents($this->tmp_dir.''.$filename, $response);
        $this->_log( "parcelAddress addresses saved to:".$this->tmp_dir.''.$filename, 6);	
        return $filename;
    }   
        
    /**                                        
     * 
     * https://webapi.foxpost.hu/api/label/A7?startPos=0
     */ 
	public function parcelLabel($barcodes=array(), $pageSize='A7', $startPos=0){
        global $wp_filesystem;
        $this->reset();
        
        $response = $this->_getResponse( "label/$pageSize?startPos=$startPos", $barcodes ); 
        if ($response==false || empty($response)) return false;
        
        $filename = time().'.pdf'; //????
        $a = $wp_filesystem->put_contents($this->tmp_dir."labels/".$filename, $response);
        return $filename;
    }                                 
    
    
    /**
     * Last status, lot of parcel
     * 
     */ 
	public function parcelsStatus($barcodes=array()){    
        if ( $this->error || empty($barcodes) ) return false;
        $this->reset();
        
        $this->_log( "parcelsStatus", 5);	
    
        $back = array();
        $traces = $this->trackParcels($barcodes);
        foreach ($traces as $trace){
            $back[ $trace->barcode ] = $trace->statuses[ 0 ]->status;
        }
        return $back;
    }
    
    /**
     *[{"status": "RECEIVE","statusDate": "2021-07-22T09:33:28.182Z","trackId": 4}]
     * 
     * 
     * 
     */ 
	public function trackParcels($barcodes=array()){
        if ( $this->error || empty($barcodes) ) return false;
        $this->reset();
        
        array_walk($barcodes, function(&$a, $b) { $a = trim((string)$a); });
        
        $this->_log( "trackParcels", 6);	
        
        $response = $this->_getResponse( "tracking/tracks", $barcodes);
        return $response;
    }
    
    
    
            
    
    /**
     * Last status one parcel
     * 
     */ 
	public function parcelStatus($barcode=''){
        $barcode = trim($barcode);
        if ( $this->error || empty($barcode) ) return false;
        $this->reset();
        
        $this->_log( "parcelStatus", 6);	
    
        $traces = $this->trackParcel($barcode); 
        if ($traces==false) return false;
        return $traces[0];
    }
    
	public function trackParcel($barcode=''){
        if ( $this->error || empty($barcode) ) return false;
        $this->reset();
        
        $this->_log( "trackParcel: $barcode", 6);	

        $response = $this->_getResponse( "tracking/$barcode" ); 
        if ($response==false) return false;
        return $response->traces;
    }
    
    public function sanitazeData($datas, $format='JSON'){
        
        $parcels = array();
        $default_data = array(
              'cod' => 0,
              'comment' => '',        
              'deliveryNote' => '',  
              'destination' => '',  
              'fragile' => false,  
              'label' => false,                           
              'recipientAddress' => '', 
              'recipientCity' => '',             
              'recipientEmail' => '',    
              'recipientName' => '',    
              'recipientPhone' => '',  
              'recipientZip' => 0,  
              'refCode' => '',    
              'size' => 'xs',  
              'uniqueBarcode' => '',  
        );       
        
        foreach($datas as $data){
            $parcel_data = array_merge($default_data, $data);
            $parcel_data['cod']              = round( $parcel_data['cod'] );
            $parcel_data['comment'] = substr( $parcel_data['comment'], 0, 50);
            $parcel_data['deliveryNote'] = substr( $parcel_data['deliveryNote'], 0, 50);
            //destination
            $parcel_data['fragile'] = (boolean) $parcel_data['fragile'];
            $parcel_data['label']   = (boolean) $parcel_data['label'];
            $parcel_data['recipientAddress'] = substr( $parcel_data['recipientAddress'], 0, 150);                
            $parcel_data['recipientCity'] = substr( $parcel_data['recipientCity'], 0, 25);              
            if ( !filter_var($parcel_data['recipientEmail'], FILTER_VALIDATE_EMAIL)) continue;  
            $parcel_data['recipientName'] = substr( $parcel_data['recipientName'], 0, 150);      
            $parcel_data['recipientPhone'] = substr( $parcel_data['recipientPhone'], 0, 13);      
            $parcel_data['recipientZip'] = intval($parcel_data['recipientZip']);                              
            $parcel_data['refCode'] = substr( $parcel_data['refCode'], 0, 30);          
            $parcel_data['size']    = strtolower($parcel_data['size']);            
            $parcel_data['uniqueBarcode'] = substr( $parcel_data['uniqueBarcode'], 0, 20);  
            
            if ( substr($parcel_data['recipientPhone'], 0, 2)=='06' ) $parcel_data['recipientPhone'] = '+36' . substr($parcel_data['recipientPhone'], 2);        
            if ($format=='CSV'){
                if ( $parcel_data['fragile']==true )    $parcel_data['fragile'] ='X';    
                if ( $parcel_data['label']==true )      $parcel_data['label']   ='X';    
            }          
            
            //print_r($parcel_data); //exit;
            if( !self::_testReqData('parcel', $parcel_data, $this->error_msg) ){
              $this->error_msg .= 'INSUFFICIENT_DATA_IN_ORDER: '.$parcel_data['refCode'].', ';
              $this->error = true;
              continue;      
            }        
               
            $parcels[] = $parcel_data;
        }
       
        return $parcels;
    }   
    
    /**
     * 
     * 
     * @param $data Array of parcels
     * @return array Assoc array ['refCode']->['clFoxId'] 
     */ 
	public function createCSV($input, $outputOrFile='inline'){
        global $wp_filesystem;
        if ($this->error) return false;
        $this->reset();
        $this->_log( "createCSV", 6);	

        $result = array();
        $datas = $this->sanitazeData( $input, 'CSV' );    
        
        //print_r($input); print_r($datas);exit;
        //Any error?
        if ( count($input)!=count($datas) || !empty($this->error_msg) ){
            //log
            $this->_log( 'Some data not fullfill.' );
            $error_ids = array_diff($input, $datas);
        }
        
        array_unshift($datas, $this->options['CSV_cols']); //add header
        //print_r( $datas );  exit;
        
        $CSV_data = array();
        foreach($datas as $rowCount=>$rowData){          //print_r( $rowData );  
          $tmp = array();
          foreach($this->options['CSV_cols'] as $key=>$value){     
            $tmp[] = (string)$rowData[$key];
            //echo "$rowCount: $key=$value, ";
            if ($rowCount>0 && $key=='refCode') $result[$rowData[$key]] = '';	
          }
          $CSV_data[] = $tmp;
        }
        //print_r($CSV_data);  exit;
        $this->_log( 'CSV line num:'.count($CSV_data) );

        $CSV = $this->writeFullQuotedCSV($CSV_data);//  exit($CSV);
        
        $this->_log( 'CSV file lenght:'.strlen($CSV) );
        
        if (!headers_sent() && $outputOrFile=='inline') {
            $filename = 'foxpost-export-'.time().'.csv';  
            
            header('Content-type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            exit($CSV);
        }else{
            $wp_filesystem->put_contents($this->tmp_dir.$outputOrFile, $CSV);
        }
        
        return $result;
       	
	}    
    
    /**
     * 
     * 
     * @param $data Array of parcels
     * @return array Assoc array ['refCode']->['clFoxId'] 
     */ 
	public function createParcel($datas){
        if ($this->error) return false;
        $this->reset();
        $this->_log( "createParcel", 6);	

        $parcels = $this->sanitazeData($datas);//    print_r($parcels);exit;
        if ($this->error) return false;
                                                      
        $response = $this->_getResponse( 'parcel', $parcels );//     var_dump($response);exit;

        if ($response==false){
            $this->error = true;
            $this->error_msg = 'FoxPost API error: '.$this->error_msg;
            return false;        
        }
        
	  

        if ($response->valid){
            $result = array();
            foreach( $response->parcels as $parcel ) $result[$parcel->refCode] = $parcel->clFoxId;	
            //return $response->parcels[0]->clFoxId;	 
            return $result;
        }else{
            $this->error = true;
            $this->error_msg .= 'FoxPost error: ';
            foreach( $response->parcels as $parcel ){
                 if ( !is_null($parcel->errors) ) foreach( $parcel->errors as $error ) {
                    $this->error_msgs[$parcel->refCode] = 'API error: '.$error->message;
                    $this->error_msg .= 'refCode#'.$parcel->refCode.': '.$error->message.'! ';
                 }           
            }
            return false;	        
        }	
	}    
    
    
    /**
     * fputcsv doesnt close beetwen quote for all field
     * 
     * 
     */ 
    public function writeFullQuotedCSV($csv_data){
        $CSV = '';
        foreach($csv_data as $row){
            $CSV .= '"'.implode ('","', $row)."\"\n";
        }        
        return $CSV;
    }    

    /**
     * Set extra user agent sting, client system info.
     *
     * @param $user_agent array Assoc array 
     * @return void
     */
    public function setUserAgent($user_agent=array())
    {
        if ( empty($user_agent) ) return;
        $ua = '';
        foreach ($user_agent as $client=>$version) $ua .= ", $client:$version";
        $this->eua = $ua;
    }
    
    /**
     * Get user agent string.
     *
     * @return string
     */
    protected function getUserAgent()
    {
        $curlValues  = curl_version();
        $curlVersion = isset($curlValues['version']) ? $curlValues['version'] : 'unknown';

        return sprintf( 'User-Agent: PHP plugin:%s, CURL:%s',  FOXPOST_PHP_API_VERSION, $curlVersion).$this->eua;
    }
           
    /**
     * Error log
     * 
     * @param $msg string Error message
     * @see _log
     */ 
	private function _err($msg)
	{
      $this->error = true;
      $this->error_msg = $msg;
      $this->_log($msg, 1);
      
      echo implode("", $this->logs);
      exit;
    }  
          
    /**
     * log message
     * 
     * @param $msg string Message
     * @param $level integer log level
     * @param $ident string  Extra ident string
	0 (EMERG)
	1 (ALERT)
	2 (CRIT)
	3 (ERR)
	4 (WARNING)
	5 (NOTICE)
	6 (INFO)
	7 (DEBUG)
	*/ 
	private function _log($msg, $level=6, $ident=''){ 
        global $wp_filesystem;
		if ( !is_string($msg) ) $msg = var_export($msg, true); 
		$log_line = gmdate('Y-m-d H:i:s');            
		if (!empty($ident)) $log_line .= " $ident";          
		$log_line .= ' '.$this->name." $msg\n";
        
		if ($this->log_level==7) echo "$log_line"; 
        $this->logs[] = $log_line; 	
		if ($level>$this->log_level) return true;  		
                
		if ( $this->logfile!==false ){
            $logfile = $this->tmp_dir . 'logs/' . gmdate('Ymd'). '.log';
            $wp_filesystem->put_contents($logfile, $log_line, FILE_APPEND);
        }
		
        $this->_deleteOldLogFiles();
        
		return true;                                                                           
	}

                    
    private function _deleteOldLogFiles(){
        $TTL = $this->log_level * 7;       
        
        $log_files = scandir($this->tmp_dir . 'logs/');
        if ($log_files==false) return 0;
        foreach($log_files as $log_file){
            if ( $log_file!='.' && $log_file!='..' && $log_file!='index.html' ){  
                $created_at = filemtime($this->tmp_dir . 'logs/' . $log_file);
                $interval = time() - $created_at;
                if ( $interval > ((int)$TTL*60*60*24) ) {
                    wp_delete_file( $this->tmp_dir . 'logs/'. $log_file );
                }
            }
        }
    }
    
	
    private function _getResponse($uri, $data=array())
	{                                                     
        $this->_log("_getResponse url: ".self::APIURL.$uri, 5);	
        $this->_log($data, 6);  
        $jsonData = wp_json_encode($data);
        
        $uris = explode("/", $uri);
                                           
        $http_headers = array(                                                                          
			'api-key'=> $this->apikey,  
            'Accept'=>$this->_getResponseType( $uris[0] ),
			'Content-Type'=>'application/json',
            'Authorization' => 'Basic ' . base64_encode( $this->username.':'.$this->password ),
        );    

	    if ( (isset($uris[1]) && in_array($uris[1], self::ENDP_METHODS['post'])) || 
             (isset($uris[0]) && in_array($uris[0], self::ENDP_METHODS['post'])) ){
            $this->_log("_getResponse POST: $jsonData", 6);
            $response = wp_remote_post(self::APIURL.$uri, [
                'user-agent'=>$this->getUserAgent(),
                'headers'=>$http_headers,
                'body'=>$jsonData,
                'sslverify'=>false,
                'timeout'=>600,
            ]); 
        }else{
            $this->_log("_getResponse GET: ".var_export($jsonData, true), 6);
            $response = wp_remote_get(self::APIURL.$uri, [
                'user-agent'=>$this->getUserAgent(),
                'headers'=>$http_headers,
                'timeout'=>600,
            ]);
        }	
		

		
        $this->_log("UA=".$this->getUserAgent(), 6);
        $decoded_response = json_decode(wp_remote_retrieve_body( $response ));
        if(isset($decoded_response->error)){
            $this->_log("_getResponse curl response is FALSE", 4);
            $this->_log($http_headers, 1);
            $this->_log('curl_error: ' . $decoded_response->error . '; curl_errno:' . $decoded_response->status, 1);
            $this->error_msg = $decoded_response->error;
            $this->error = true;
            return false;
		}
		if ( is_wp_error($response)) 
		{
            $this->_log("_getResponse curl response is FALSE", 4);
            $this->_log($http_headers, 1);
            $this->_log('curl_error: ' . $response->get_error_message() . '; curl_errno:' . $response->get_error_code(), 1);
            $this->error_msg =  $response->get_error_message();
            $this->error = true;
            return false;
		}
        if ( $uris[0]!='label' ) $this->_log("_getResponse response: ".wp_remote_retrieve_body( $response ), 7);
        
         
        $httpcode = wp_remote_retrieve_response_code($response);
        $this->_log("_getResponse HTTP code: ".$httpcode, 6);
		switch ( (int)$httpcode ) {
				case 201:
				case 200:
                    if ('label'!=$uris[0] && 'address'!=$uris[0]) $response = $decoded_response ;
					break;
				case 401:
                    $this->error = true;
                    $this->error_msg = 'Unauthorized';
                    $response = false;
					break;
				case 400:
                    $this->error = true;
                    $response = json_decode($response);
                    $this->error_msg = $response->error;//'NOT_VALID'; //PARCEL_NOT_VALID
                    $response = false;
					break;
                
				default:
                    $this->error = true;
                    $this->error_msg = 'ERROR'; //405 Method Not Allowed - HTTP - MDN Web Docs  , 415  
                    $response = false;
					break;
		}          
        if ( $uris[0]!='label' ) $this->_log("_getResponse response: ".var_export($response, true), 6);
        		
		return $decoded_response ?? wp_remote_retrieve_body( $response );
	}   



    /**
     * @return bool
     */
    private static function _isCurlAvailable()
    {
        return function_exists('curl_init');
    }
    
    private function _getResponseType($ep){
        foreach (self::ENDP_RESP_TYPE as $type=>$eps){ 
            if ( in_array($ep, $eps) ) return "application/$type";
        }
        return 'application/json';
    }
    
    private static function _testReqData($ep, $data, &$error){
        foreach (self::ENDP_DATA_REQ[$ep] as $key){ 
            if ( !isset($data[$key]) || empty($data[$key]) ){
                $error .= "$key empty, ";
                return false;
            } 
        }
        return true;
    }
    
    
}

?>