<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly 
/**
 * Handle Action Scheduler integration.
 *
 * @link       https://foxpost.hu
 * @since      1.0.0
 *
 * @package    Foxpost_Wc_Shipping
 * @subpackage Foxpost_Wc_Shipping/includes
 */
 
require_once (dirname(plugin_dir_path(__FILE__)))
            . '/includes/FoxPostApi.php';
            
/**
 * Action Scheduler integration.
 *
 * 
 *
 * @since      1.0.0
 * @package    Foxpost_Wc_Shipping
 * @subpackage Foxpost_Wc_Shipping/includes
 * @author     Foxpost.hu <b2chelpdesk@foxpost.hu>
 */


if ( ! class_exists( 'Foxpost_Wc_Shipping_Scheduler', false ) ) :

	class Foxpost_Wc_Shipping_Scheduler {
        private $config = array(); 
        private $logger = null; 
                   
    	public function __construct() {    
            $this->config = self::loadConfig();
            if (is_null($this->logger)) $this->logger = new Foxpost_Wc_Shipping_Logger();   
    	}    
 
        /**
         * Load configuration info.
         * 
         * 
         */    
        public static function loadConfig() {
            $c = plugin_dir_path( dirname( __FILE__ ) ) .'config.php';
            if (file_exists($c)) return include ($c);
            return array();  
        }

        /**
         * Delete automation, call from deactivator.
         * 
         * 
         */ 
 		public static function deinit() {
			remove_action( 'foxpost_wc_shipping_parcel_update', array( __CLASS__, 'get_parcel_state' ) );
            WC()->queue()->cancel_all( 'foxpost_wc_shipping_parcel_update' );
		}

        /**
         * Setup daily scheduled action to refresh databases
         * 
         * 
         */          
		public static function init() {
			add_action( 'foxpost_wc_shipping_parcel_update', array( __CLASS__, 'get_parcel_state' ) );
		}

        /**
         * Install parcel check automation.
         * 
         * 
         */ 
		public static function schedule_actions() {		
            $logger = new Foxpost_Wc_Shipping_Logger();   
            $logger->log_d('Foxpost_Wc_Shipping_Scheduler::schedule_actions');   
                
                		
            //only load it ONCE
            $exist = WC()->queue()->search( array('hook'=>'foxpost_wc_shipping_parcel_update','status'=>ActionScheduler_Store::STATUS_PENDING), 'ids' );
            //var_dump($exist);
            if (is_array($exist) && !empty($exist) ) return;
		
                
            //One time
			//WC()->queue()->add( 'foxpost_wc_shipping_parcel_update', array(), 'foxpost_wc_shipping' );

			//Setup scheduled actions         
			//WC()->queue()->schedule_recurring( time()+(MINUTE_IN_SECONDS*15), (MINUTE_IN_SECONDS*15), 'foxpost_wc_shipping_parcel_update', array(), 'foxpost_wc_shipping', true );
			$first_run = new DateTime();
            $first_run->modify('+1 day');
            $first_run->setTime(4,30);
            WC()->queue()->schedule_recurring( $first_run->getTimestamp(), DAY_IN_SECONDS, 'foxpost_wc_shipping_parcel_update', array(), 'foxpost_wc_shipping', true );
                      
            $logger->log_i('Foxpost_Wc_Shipping_Scheduler::schedule_actions automation installed');   
		}

        /**
         * Get "FoxpostParcelStatusToOrderStatus" order from database and call FoxPostApi->parcelsStatus (tracking/tracks endpoint)
         * config.php, DB
         * 
         * 
         */ 
		public static function get_parcel_state() {
            global  $wp_filesystem;
            $static = !(isset($this) && $this instanceof self);
            $log_me_pls = ($static) ? new Foxpost_Wc_Shipping_Logger()  : $this->logger;
            $config     = ($static) ? self::loadConfig()                : $this->config; 
            
            
            $log_me_pls->log_i('Foxpost_Wc_Shipping_Scheduler::get_parcel_state');  
            
			$need_sync = self::check_if_sync_needed();
			if(!$need_sync){
                return false;
                $log_me_pls->log_d('Foxpost_Wc_Shipping_Scheduler::check_if_sync_needed is FALSE'); 
            } 
            
            if ( defined('WP_DEBUG') && constant("WP_DEBUG")==true ){
                $log_me_pls->log_d('WP_DEBUG is on');   
                $debug_msg = 'Foxpost_Wc_Shipping_Scheduler::get_parcel_state runned';
                $path = wp_upload_dir( null, false ); 
                $wp_filesystem->put_contents($path['basedir'].DIRECTORY_SEPARATOR.'foxpost-wc-shipping'.DIRECTORY_SEPARATOR.'logs'.DIRECTORY_SEPARATOR.'cron.log', gmdate('Y-m-d H:i:s')." $debug_msg\n", FILE_APPEND);  
            }

            if ( !isset($config['FoxpostParcelStatusToOrderStatus']['READY']) || !is_array($config['FoxpostParcelStatusToOrderStatus']['READY']) ){
                $log_me_pls->log_e('Config error! In config.php FoxpostParcelStatusToOrderStatus.READY is missing! Sync not available! Exit, die, halt.');  
                return false;
            }
      
            global $wpdb; 
            // $SQL_array = "'".implode("','", $config['FoxpostParcelStatusToOrderStatus']['READY'])."'";
            $sql_array_values = esc_sql($config['FoxpostParcelStatusToOrderStatus']['READY']);
            $SQL_array = implode(', ',$sql_array_values);
            // $SQL = $wpdb->prepare(
            //         "SELECT pm1.meta_value AS foxpostId, pm1.post_id AS orderId
            //             FROM {$wpdb->prefix}posts AS p 
            //             JOIN {$wpdb->prefix}postmeta pm1 ON p.ID=pm1.post_id
            //             JOIN {$wpdb->prefix}postmeta pm2 ON p.ID=pm2.post_id
            //             WHERE 
            //             pm1.meta_key=%s AND pm1.meta_value IS NOT NULL AND pm1.meta_value<>'' AND 
            //             pm2.meta_key=%s AND (pm2.meta_value NOT IN (%s) OR pm2.meta_value IS NULL)   
            //         ",
            //         'foxpost_wc_shipping_barcode',
            //         '_foxpost_wc_shipping_parcel_status',
            //         $SQL_array,
            //      );

            // $log_me_pls->log_d( $SQL );    
           
            $query_args = [
                'limit' => -1,
                'meta_query' => [
                    'relation' => 'AND',
                    [
                        'key'     => 'foxpost_wc_shipping_barcode',
                        'compare' => 'EXISTS',
                    ],
                    [
                        'key'     => 'foxpost_wc_shipping_barcode',
                        'value' => '',
                        'compare' => '!='
                    ],
                    [
                        'key'     => '_foxpost_wc_shipping_parcel_status',
                        'value'   => $SQL_array,
                        'compare' => 'NOT IN',
                    ],
                    [
                        'key'     => '_foxpost_wc_shipping_parcel_status',
                        'compare' => 'EXISTS',
                    ],
                ]
            ];
            $ordersQueryResults = (new WC_Order_Query($query_args))->get_orders();
            $orders = [];
            foreach($ordersQueryResults as $order){
                $clFox = $order->get_meta('foxpost_wc_shipping_barcode', true);
                $orders[$clFox] = [
                    'foxpostId'=> $clFox,
                    'orderId'=>$order->get_id(),
                ];
            }
            if ( empty($orders) ){
                $log_me_pls->log_i( 'No relevant order' ); 
                return false;
            }            
            
            $log_me_pls->log_i( sprintf('checked orders(%s): %s', count($orders), var_export($orders, true) ) );   

      
            $parcels = array_keys($orders);       //FoxId=>OrderId
            $log_me_pls->log_i( sprintf('checked barcodes(%s): %s', count($parcels), implode(',',$parcels) ) );   

         
            //include API 
            $loglevel = ( get_option('foxpost_wc_shipping_api_log_level')!=false ) ? get_option('foxpost_wc_shipping_api_log_level') : 3;       
            $FoxPostApi = new FoxPostApi($loglevel);  

            $p = wp_upload_dir( null, false );             
            $FoxPostApi->setTmpDir( $p['basedir'].DIRECTORY_SEPARATOR.'foxpost-wc-shipping'.DIRECTORY_SEPARATOR );
                    
            global $woocommerce, $wp_version; 
            $FoxPostApi->setUserAgent(array('WP'=>$wp_version, 'WooCommerce'=>$woocommerce->version));   
                  
            $FoxPostApi->setCred( get_option('foxpost_wc_shipping_username'), get_option('foxpost_wc_shipping_password'), get_option('foxpost_wc_shipping_apikey') );  
            $parcelStatus = $FoxPostApi->parcelsStatus($parcels); //var_dump($parcelStatus);   var_dump();
        
    
            //log        
            if($FoxPostApi->error==true) {
                $log_me_pls->log_e($parcelStatus);  
                $log_me_pls->log_e($FoxPostApi); 
                return 0;
            }  
     
             
            //update
            $translations = Foxpost_Wc_Shipping::loadConfig()['FoxpostParcelStatusToOrderStatusTranslated'];
            $updated_orders = array();
            foreach ($parcelStatus as $key=>$status){
                $translated_status =  $translations[$key];
                
                $order = wc_get_order( $orders[$key]['orderId'] );   
                $order->update_meta_data('_foxpost_wc_shipping_parcel_status', $status);
                $order->update_meta_data('_foxpost_wc_shipping_parcel_status_name', $translated_status);
                $order->save();
                
                $updated_orders[] = $orders[$key]['orderId'];
                
                //Status change
                $config = Foxpost_Wc_Shipping::loadConfig()['FoxpostParcelStatusToOrderStatus'];
                $full_config = array();
                foreach ($config as $nameORid=>$parcelstate){
                    if (is_array($parcelstate)) foreach ($parcelstate as $v)    $full_config[$v] = $nameORid;
                    else                                                        $full_config[$parcelstate] = $parcelstate;
                }
          
                
                $trigger = get_option('foxpost_wc_shipping_parcel_vs_order', array());
                
                if ( isset($full_config[$status]) && in_array($full_config[$status], $trigger['foxpost']) ) {
                    //OMG it is very-very-very easy
                    $newOrderStatus = $trigger['wc'][ array_search($full_config[$status], $trigger['foxpost']) ];
                    //exit("*parcel status: $status, order status:$newOrderStatus*");   
                    
                    $order = wc_get_order( $orders[$key]['orderId']);
                    // Add the note
                    /* translators: %s: Text of status */
                    $note = sprintf( __( "The parcel state changed to: %s" ), $translations[$key], 'foxpost-shipping-method' );
                    $order->add_order_note( $note );
                    
                    //Update order status
                    $order->update_status($newOrderStatus);
                    
                    $order->save();


                    $log_me_pls->log_i( 'Updated order #'.$orders[$key]['orderId'].". New parcel status: $status, new order status: $newOrderStatus" ); 
                }                    
                //Status change                            
                if ( get_option('foxpost_wc_shipping_parcel_received_state', 'no-action') !== 'no-action' && 
                     in_array($status, $config['FoxpostParcelStatusEndStatuses'])
                    ) {
    
                    $order = wc_get_order( $orders[$key]['orderId'] );
                    $newOrderStatus = get_option('foxpost_wc_shipping_parcel_received_state');
                    $order->update_status($newOrderStatus);
                    
                    // Add the note
                    /* translators: %s: order new status */
                    $note = sprintf( __( "The parcel state changed to: %s" ), $newOrderStatus , 'foxpost-shipping-method');
                    $order->add_order_note( $note );

                    $order->save();
                    $log_me_pls->log_i( 'Updated order #'.$orders[$key]['orderId'].' status to '.$newOrderStatus ); 
                }              
              
              $order = wc_get_order( $orders[$key]['orderId'] );   
              $order->update_meta_data('_foxpost_wc_shipping_parcel_status_name', $status);
              $order->save();
            }
          
            $lastsyncfile = Foxpost_Wc_Shipping::getDir('basedir').'logs/last_api.sync'; //   exit($lastsyncfile);
            $wp_filesystem->put_contents($lastsyncfile, time());
            
            $log_me_pls->log_i( 'Updated orders: '.implode(',',array_values($updated_orders)) ); 

            return count($parcelStatus);              
		}


		public static function check_if_sync_needed() {
			if ( (int)Foxpost_Wc_Shipping::getLastParcelSyncTime() + (Foxpost_Wc_Shipping::SYNC_MIN_MINUTE*60) < time() ) return true;
            return false;
		}
           


	}//end Foxpost_Wc_Shipping_Scheduler

    Foxpost_Wc_Shipping_Scheduler::init();

endif;
